#ifndef LLF_HASH_H
#define LLF_HASH_H

/*
* All the includes that are needed for code using this module to
* Compile correctly should be #included here.
*/

#include "CE2_public.h"
#include "CE2_HASH.h"

#ifdef __cplusplus
extern "C"
{
#endif

/*
* Creation date : Fri Mar 02 10:33:36 2007
* Last modified : %modify_time%
*/
/** @file
* \brief This file contains declaration of low level function
* provides a HASH function for processing one buffer of data,
* which worked with LibTomCrypt.
* 
* \version LLF_HASH.h#1:csrc:1
* \author Pavel Sasunkevich
* \remarks Copyright (C) 2007 by Discretix Technologies Ltd.
* All Rights reserved
*/

/************************ Defines *****************************/
/************************ Enums *******************************/
/************************ Typedefs ****************************/
/************************ Structs *****************************/
/************************ Public Variables ********************/
/************************ Public Functions ********************/

/**
****************************************************************
* Function Name: 
*  LLF_HASH_Init
*
* Inputs:
* @param ContextID_ptr [in] - A pointer to the HASH context buffer allocated by the user 
*                      that is used for the HASH machine operation.
* @param OperationMode [in] - The operation mode:
*                      CE2_HASH_SHA1_mode, CE2_HASH_SHA224_mode, CE2_HASH_SHA256_mode,
*                      CE2_HASH_SHA384_mode, CE2_HASH_SHA512_mode, CE2_HASH_MD5_mode.
*
* Outputs: @returns \b
*  CE2Error_t  
*  - CE2_OK - On success
*  - Otherwise - error code:
*   - CE2_HASH_INVALID_USER_CONTEXT_POINTER_ERROR
*   - CE2_HASH_ILLEGAL_HASH_OP_MODE_ERROR
*
* \brief \b
* Description:
*  This function initializes the HASH machine on the CE2 level.
* This function allocates and initializes the HASH Context.
* The function receives as input a pointer to store the context handle to the HASH Context,
* and it initializes the HASH Context with the cryptographic attributes that are needed for 
* the HASH block operation (initializes H's value for the HASH algorithm).
*
* \b
* Algorithm:
*  -# Initializes HASH context with using inputted data.
***************************************************************/
CE2Error_t LLF_HASH_Init(CE2_HASHUserContext_t		*ContextID_ptr,
						 CE2_HASH_OperationMode_t	OperationMode);

/**
****************************************************************
* Function Name: 
*  LLF_HASH_Update
*
* Inputs:
* @param ContextID_ptr [in] - A pointer to the HASH context buffer allocated by the user 
*                      that is used for the HASH machine operation.
* @param DataIn_ptr [in] - A pointer to the buffer that stores the data to be hashed;
* @param DataSize [in] - The size of the data to be hashed in bytes;
*
* Outputs: @returns \b
*  CE2Error_t  
*  - CE2_OK - On success
*  - Otherwise - error code:
*   - CE2_HASH_INVALID_USER_CONTEXT_POINTER_ERROR
*   - CE2_HASH_INVALID_DATA_IN_PTR_ERROR
*
* \brief \b
* Description:
*  This function process a block of data via the HASH Hardware/Software.
* The function receives as input an handle to the HASH Context that was previously 
* initialized by a CE2_HASH_Init function or by another call to a CE2_HASH_Update function. \n
* The function sets the last H's value that was stored in 
* the CE2 HASH context, and then processes the data block. 
* At the end of the process, the function stores in the HASH context the H's value, 
* together with the cryptographic attributes that are needed for 
* the HASH block operation (initialize H's value for the HASH algorithm).
* This function is used in cases where the data is not all arranged in one 
* continuous buffer. \n
*
* \b
* Algorithm:
*  -# Update HASH using inputted data with using LibTomCrypt.
***************************************************************/
CE2Error_t LLF_HASH_Update(CE2_HASHUserContext_t	*ContextID_ptr,
						   DxUint8_t				*DataIn_ptr,
						   DxUint32_t				DataInSize);

/**
****************************************************************
* Function Name: 
*  LLF_HASH_Finish
*
* Inputs:
* @param ContextID_ptr [in] - A pointer to the HASH context buffer allocated by the user 
*                      that is used for the HASH machine operation.
* @param HashResultBuff [out] - A pointer to the target buffer where the 
*                  HASH result stored in the context is loaded to.
*
* Outputs: @returns \b
*  CE2Error_t  
*  - CE2_OK - On success
*  - Otherwise - error code:
*   - CE2_HASH_INVALID_USER_CONTEXT_POINTER_ERROR
*   - CE2_HASH_INVALID_HASH_RESULT_BUFF_PTR_ERROR
*
* \brief \b
* Description:
*   This function finalizes the hashing process of data block.
* The function receives as input an handle to the HASH Context , that was initialized before
* by an CE2_HASH_Init function or by CE2_HASH_Update function.
* The function "adds" an header to the data block as the specific hash standard 
* specifics, then it loads the engine and reads the final message digest.
*
* \b
* Algorithm:
*  -# Finish HASH calculation and copy it into HASH result buffer
*     with using LibTomCrypt.
***************************************************************/
CE2Error_t LLF_HASH_Finish(CE2_HASHUserContext_t	*ContextID_ptr,
						   CE2_HASH_Result_t		HashResultBuff);

/**
****************************************************************
* Function Name: 
*  LLF_HASH_Free
*
* Inputs:
* @param ContextID_ptr [in] - A pointer to the HASH context buffer allocated by the user 
*                      that is used for the HASH machine operation.
*
* Outputs: @returns \b
*  CE2Error_t  
*  - CE2_OK - On success
*  - Otherwise - error code:
*   - CE2_HASH_INVALID_USER_CONTEXT_POINTER_ERROR
*
* \brief \b
* Description:
*  This function is a utility function that frees the context if the operation has failed.
*
*        The function executes the following major steps:
*
*        1. Checks the validity of all of the inputs of the function. 
*           If one of the received parameters is not valid it shall return an error.
*           The main checks that are run over the received parameters:
*           - verifying that the context pointer is not DX_NULL (*ContextID_ptr). 
*        2. Clears the user's context.
*        3. Exits the handler with the OK code.
*
* \b
* Algorithm:
*  -# Clear HASH context.
***************************************************************/
CE2Error_t LLF_HASH_Free(CE2_HASHUserContext_t *ContextID_ptr);

/**
****************************************************************
* Function Name: 
*  LLF_HASH
*
* Inputs:
* @param OperationMode [in] - The operation mode :
*                  SHA1, SHA224, SHA256, SHA384, SHA512, MD5;
* @param DataIn_ptr [in] - A pointer to the buffer that stores the data to be hashed;
* @param DataSize [in] - The size of the data to be hashed in bytes;
* @param HashResultBuff [out] - A pointer to the target buffer where the 
*                  HASH result stored in the context is loaded to.
*
* Outputs: @returns \b
*  CE2Error_t  
*  - CE2_OK - On success
*  - Otherwise - error code.
*
* \brief \b
* Description:
*   The function allocates an internal HASH Context, and initializes the 
* HASH Context with the cryptographic attributes that are needed for 
* the HASH block operation (initialize H's value for the HASH algorithm).
* Next the function loads the engine with the initializing values, 
* and then processes the data block, calculating the hash.
* Finally, the function return the message digest of the data buffer .
*
* \b
* Algorithm:
*  -# Initialize HASH machine;
*  -# Perform hash operation.
***************************************************************/
CE2Error_t LLF_HASH  ( CE2_HASH_OperationMode_t  OperationMode,	/* in */
                       DxUint8_t                 *DataIn_ptr,	/* in */
                       DxUint32_t                DataSize,		/* in */
                       CE2_HASH_Result_t         HashResultBuff );/* out */

#ifdef __cplusplus
}
#endif

#endif /* LLF_HASH_H */
